;-----------------------------------------------------------------------------
; RELAY DRIVER REV E2 FOR USE WITH DPB2 REV E
; it will not work with DPB2 REV D
; (The rev level is on the silkscreen on your board).
;
; (C) Copyright 1994 Systronix, Inc All Rights Reserved
;		Systronix, Inc. Salt Lake City, Utah, USA
;		TEL: 801-534-1017  FAX:801-534-1019  BBS:801-487-2778
;
; Owners of Systronix products are hereby entitled to use, modify, and copy
; this sample program.  You may incorporate all or part of this program in 
; your own commercial products provided they are not simply copies of 
; Systronix products and do not compete directly with Systronix products.  
; This program is provided by Systronix free of charge and may not be resold.
;
; This program is intended for those of you who are enjoying the benefits of
; the new DPB2 REVE board from Systronix/Intellix.  This card is a very
; easy to use development system for the Dallas DS5000/1/2 DS2250/1/2 and -T
; suffix soft microcontrollers.  If you'd like more information, please call
; or FAX us at the numbers above.
;
; REVISION HISTORY -----------------------------------------------------------
;
; 96 Sep 23 BAB	E2, renamed file to rly_e2.inc
;		Adding improvements and optimizing. This driver is less than
;		half the size of the rly_e.inc which it replaces.
;
; 01/24/94	BAB	Start, derived from RELAY1.INC for DPB2 rev D
;
;
; NOTES AND COMMENTS ---------------------------------------------------------
;
; Allegro UCN5800A
;
; The relay driver has 4 open collector outputs, addressed on the data bus
; as P0.0-3.  Strobe the data with P1.1.
; The data latch looks like a 373 family device: when the strobe is high,
; data flows through transparently, and when strobe is low the data is held.
;
; On power up P1.1 is high, so to turn off the ports, drive P0 low and then
; drive P1.1 low. The driver does have a power on clear.  We drive its OE(L) 
; with a port pin P1.2 which powers up HIGH, disabling the driver until we 
; deliberately enable it.
;
; We use relay 1 (data bit 0) to drive an inexpensive piezo buzzer, available
; from Systronix to click and beep.  It clicks once per second in the 
; metronome subroutine.
;
; INSTRUCTIONS ---------------------------------------------------------------
;
; The relay driver is a latch with an output enable. Therefore the data remains
; intact if the output is enabled or disabled, until you change the data.
;
; Variables, unsigned chars (8-bit) ------------------------------------------
;	relay_status	one bit for each relay, bit=1 if ON
;			Can be 00H-0FH, a value of 0FH means all relay
;			data is set to 1 (the on state). This variable
;			lets you change more than one relay output at a time.
;	relay_number	"user" relay number 1-4, this is the bit which will change
;			values greater than 4 are ignored. So you can only change
;			one bit at a time using this variable. Value of 0 means
;			no change.
;
; BASIC-callable subroutines -------------------------------------------------
;	relay_init	clears relay data, disables outputs, clears relay_status
;	relay_enable	enables the relay outputs, relay data unaffected
;	relay_disable	disables the relay outputs, relay data unaffected
;	relay_turn1_on	turns on one relay, identified by relay_number
;			updates the value of relay_status
;	relay_turn1_off	turns one relay, identified by relay_number
;			updates the value of relay_status
;	relay_toggle1	toggles one relay, identified by relay_number
;			updates the value of relay_status
;	relay_update	sets all relay outputs to the value of relay_status
;	metronome_1	clicks relay 1 each time called, intended to have a buzzer
;			on relay 1
;	beep_1		beeps a buzzer on relay 1
;	flash_2		flashes the LED backlight of the LCD, if the backlight's
;			ground terminal is pulled low by relay 2
;
;-----------------------------------------------------------------------------

	goto relay_inc_end         		; protect against include in program body

	unsigned char relay_status		; one bit for each relay, bit=1 if ON
	unsigned char relay_number		; "user" relay number 1-4

	; uncomment these if your BASIC program doesn't access these variables.
	; (This otherwise unneeded initialization uses the variables and therefore
	; forces BCI51 to allocate memory for these variables)
;	relay_status = 0			; force BCI51 to allocate these
; 	relay_number = 0			

;-----------------------------------------------------------------------------
;	RELAY INIT
;
; Call this at the start of your program before you try to use the relay
; outputs. This disables the outputs and then clears the relay latch.
; Clears the value of the BASIC variable relay_status
; Used here: A, DPTR
;
relay_init:
#ASM
	setb	P1.2					; disable relay outputs
	mov     P0,#000H				; shut off relay outputs
	setb    P1.1					; strobe in data, force P1.1 high, then low
	clr     P1.1
	mov	DPTR, #__RELAY_STATUS
	clr	A
	movx	@DPTR, A				; clear relay_status
#ASM_END
	return

; relay_set_dat as of 96 Sep 23 is a multiply-nested if-else, this will generate
; much more code than we really need.
;  We can do a lot better in assy code.

;------------------------------------------------------------------------------
; RELAY_SET_DAT
;
; Call this routine with relay_number set to the relay you want to change,
; 1-4. relay_number of 0 means no change.
; All that happens here is that one bit of relay_dat is set to correspond
; to the relay_number - i.e., relay_number = 3 sets relay_dat bit 3
; (4th bit). Relay number 1 sets bit 0, the 1st bit of relay_dat.
;
; This routine leaves the value in the accumulator for
; subsequent use by the caller.
; Used: DPTR, A, B, PSW (carry)

#ASM
_relay_set_dat:
	mov	DPTR, #__RELAY_NUMBER		; contains value 1-4
	movx	A, @DPTR			; retrieve it
	jz	_relay_setx			; if acc = 0, no changes
	cjne	A, #5, _relay_seta		; C set if acc < 5
_relay_seta:
	jc	_relay_setb			; jumps if number OK
	clr	A				; fall thru to here if number not OK, clear it
	jnc	_relay_setx			; relay_number >=5, invalid so ignore it

_relay_setb:
	mov	B, A				; relay number now in B reg
	clr	A
	setb	ACC.7				; set up to rotate this into acc.0
				
_relay_setc:	
	rl	A				; rotate the 1 left
	djnz	B, _relay_setc			; do it relay_number times
	; now acc contains 8,4,2, or 1 to set the relay bit = relay_number

_relay_setx:
	; we're here with either acc = relay bit set or acc=0 if no change or bad value
	ret
#ASM_END

	

;-----------------------------------------------------------------------------
;	RELAY ENABLE
; This short routine enables the relay outputs
;
relay_enable:
#ASM
	clr	P1.2					; drive low to turn on
#ASM_END
	return
	
;-----------------------------------------------------------------------------
;	RELAY DISABLE
; This short routine disables the relay outputs. Regardless of the relay data,
; no outputs will be driven low. The data is not changed, so that calls to enable
; or disable the relays will toggle any active outputs on and off.
;
relay_disable:
#ASM
	setb	P1.2					; drive high to turn off
#ASM_END
	return

;-----------------------------------------------------------------------------
;	RELAY TURN1 ON
; Turn on any single relay bit.
; The bit to be changed is assumed to be in the BASIC variable RELAY_NUMBER.
; This routine updates the BASIC variable RELAY_STATUS.
; Calls other routines.
; Used here: DPTR, A, B
;
relay_turn1_on:
#ASM
	lcall 	_relay_set_dat				; acc holds relay_dat
	mov	B, A					; copy to B reg
	mov	DPTR, #__RELAY_STATUS
	movx	A, @DPTR				; current status in acc
	orl	A, B					; OR in the new
	movx	@DPTR, A				; save new status	
	lcall	_relay_update				; update relay bits
#ASM_END
	return

;-----------------------------------------------------------------------------
;	RELAY TURN1 OFF
; Turn off any single relay bit.
; The bit to be changed is assumed to be in the BASIC variable RELAY_NUMBER.
; This routine updates the BASIC variable RELAY_STATUS.
; Calls other routines.
; Used here: DPTR, A, B

relay_turn1_off:
#ASM
	lcall 	_relay_set_dat				; acc holds relay_dat
	cpl	A					; NOT of relay_dat
	mov	B, A					; copy to B reg
	mov	DPTR, #__RELAY_STATUS
	movx	A, @DPTR				; current status in acc
	anl	A, B					; clear the bit w/our mask
	movx	@DPTR, A				; save new status	
	lcall	_relay_update				; update relay bits
#ASM_END
	return

;-----------------------------------------------------------------------------
;	RELAY TOGGLE 1
; Toggle any single relay bit.
; The bit to be changed is assumed to be in the BASIC variable RELAY_NUMBER.
; This routine updates the BASIC variable RELAY_STATUS.
; Calls other routines.
; Used here: DPTR, A, B
relay_toggle1:
#ASM
	lcall 	_relay_set_dat				; acc holds relay_dat
	mov	B, A					; copy to B reg
	mov	DPTR, #__RELAY_STATUS
	movx	A, @DPTR				; current status in acc
	xrl	A, B					; clear the bit w/our mask
	movx	@DPTR, A				; save new status	
	lcall	_relay_update				; update relay bits
#ASM_END
	return

;-----------------------------------------------------------------------------
;	RELAY UPDATE
; Drives relay outputs to reflect the current value of RELAY_STATUS
; Used: DPTR, A
;
; If you want to change multiple relay outputs, you can set relay_status
; and then call relay_update.

relay_update:
#ASM
_relay_update:
	mov     DPTR, #__RELAY_STATUS    	; addr in DPTR
	movx    A, @DPTR                 	; RELAY_STATUS in Port 0
	push	IE				; save interrupt status
	clr	EX1				; prevent ONEX1 from interrupting
	mov     P0, A
	
	setb    P1.1				; strobe in data, force P1.1 high, then low
	clr     P1.1
	clr	P1.2				; be sure relay outputs are enabled
	pop	IE				; restore interrupts
	ret
#ASM_END


;-----------------------------------------------------------------------------
;	METRONOME
; Assumes a buzzer is on relay output 1, driven by data bit 0.
; A click each time this is called, like a metronome.
; Or the drip of Chinese water torture!
; Used: DPTR, A, R0

metronome_1:
#ASM
	mov     DPTR,#__RELAY_STATUS		; addr in DPTR
	movx    A,@DPTR				; RELAY_STATUS in Port 0
	mov     P0,A
	setb	P0.0			        ; set relay 1 to on
	; strobe in data, force P1.1 high, then low
	setb    P1.1
	clr     P1.1
	mov     R0,#0FFH			; one djnz = 2 cycles (~ 2 usec)
	djnz    R0,$                    	; tone duration
	clr     P0.0                    	; turn relay 1 off
	; strobe in data, force P1.1 high, then low
	setb    P1.1
	clr     P1.1
#ASM_END
	return


;-----------------------------------------------------------------------------
;	BEEP 1
; Assumes a buzzer is on relay output 1, driven by data bit 0.
; Makes a beep, exact length depends on MCU xtal.
; Used: DPTR, A, R0
beep_1:
#ASM
	mov     DPTR,#__RELAY_STATUS    	; addr in DPTR
	movx    A,@DPTR                 	; RELAY_STATUS in Port 0
	mov     P0,A
	setb	P0.0			        ; set relay 1 to on
	; strobe in data, force P1.1 high, then low
	setb    P1.1
	clr     P1.1

	; set tone duration one, 1 djnz = 2 cycles (~ 2 usec)
	mov     R1,#02FH
beep_loop:
	mov     R0,#0FFH
	djnz    R0,$ 
	djnz    R1,beep_loop

	clr     P0.0                    	; turn relay 1 off
	; strobe in data, force P1.1 high, then low
	setb    P1.1
	clr     P1.1
#ASM_END
	return

;-----------------------------------------------------------------------------
;	FLASH LCD BACKLIGHT
; flash backlight iff LED backlight is tied to relay output #2
; Used: DPTR, A, R0

flash_2:
#ASM
	mov     DPTR,#__RELAY_STATUS 		; addr in DPTR
	movx    A,@DPTR				; RELAY_STATUS in Port 0
	mov     P0,A
	setb	P0.1			        ; set relay 2 on
	setb    P1.1				; strobe in data, force P1.1 high, then low
	clr     P1.1
	mov     R0,#0FFH                	; one djnz = 2 cycles (~ 2 usec)
	djnz    R0,$                    	; tone duration
	mov     R0,#0FFH                	; one djnz = 2 cycles (~ 2 usec)
	djnz    R0,$                    	; tone duration
	clr     P0.0                    	; turn relay 2 off
	; strobe in data, force P1.1 high, then low
	setb    P1.1
	clr     P1.1
#ASM_END
	return


relay_inc_end:                  		; protect against include in program body